port module Main exposing (..)

import Browser
import Browser.Events
import Html exposing (..)
import Html.Attributes exposing (..)
import Html.Events exposing (..)
import Json.Decode as D
import Time



-- MAIN


main : Program () Model Msg
main =
    Browser.element
        { init = init
        , view = view
        , update = update
        , subscriptions = subscriptions
        }



-- PORTS


port messageReceiver : (String -> msg) -> Sub msg


port pause : (Bool -> msg) -> Sub msg



-- MODEL


type alias Model =
    { messages : List String
    , paused : Bool
    }


init : () -> ( Model, Cmd Msg )
init flags =
    ( { messages = [], paused = False }
    , Cmd.none
    )



-- UPDATE


type Msg
    = FromPort String
    | Tick Time.Posix
    | Pause Bool


update : Msg -> Model -> ( Model, Cmd Msg )
update msg model =
    Debug.log "update" <|
        ( case msg of
            FromPort message ->
                { model | messages = message :: model.messages }

            Tick posix ->
                { model | messages = String.fromInt (Time.posixToMillis posix) :: model.messages }

            Pause True ->
                { model | paused = True, messages = "paused" :: model.messages }

            Pause False ->
                { model | paused = False, messages = "unpaused" :: model.messages }
        , Cmd.none
        )


subscriptions : Model -> Sub Msg
subscriptions { paused } =
    [ if paused then
        Sub.none

      else
        -- I would not use ports and the next line together, but enjoy
        -- Browser.Events.onAnimationFrame Tick
        messageReceiver FromPort
    , pause Pause
    ]
        |> Sub.batch


view : Model -> Html Msg
view model =
    div []
        [ h1 [] [ text "Pausing subscriptions" ]
        , p []
            [ text "Pausing works just as well for ports as it does for "
            , a [ href "https://package.elm-lang.org/packages/elm/browser/latest/Browser-Events#onAnimationFrame" ] [ text "Browser.events.onAnimationFrame" ]
            , br [] []
            , text "One more thing: Open the browser console to see the expandable `Debug.log` output."
            ]
        , button [ onClick <| Pause <| not model.paused ] [ text "Pause" ]
        , List.take 15 model.messages
            |> List.map (\msg -> li [] [ text msg ])
            |> ul []
        ]
