module Main exposing (main)

import Browser
import Html exposing (Html, button, div, h1, node, text)
import Html.Attributes exposing (attribute, class, disabled, property)
import Html.Events exposing (on, onClick)
import Json.Decode
import Json.Encode


type Model
    = Choose
    | CountDown Seconds
    | CountUp Seconds


type Msg
    = StartCount Direction Seconds
    | StopCount Direction


type Direction
    = Up
    | Down


type alias Seconds =
    Float


update : Msg -> Model -> Model
update msg model =
    case msg of
        StartCount Down seconds ->
            CountDown seconds

        StartCount Up seconds ->
            CountUp seconds

        StopCount direction ->
            Choose


view : Model -> Html Msg
view model =
    div []
        [ h1 [] [ text "Custom Element example" ]
        , button
            [ onClick <| StartCount Down 3.0
            , disabled <| model /= Choose
            ]
            [ text "Count down 3s" ]
        , button
            [ onClick <| StartCount Up 4.0
            , disabled <| model /= Choose
            ]
            [ text "Count up to 4s" ]
        , counter <|
            case model of
                Choose ->
                    []

                CountDown seconds ->
                    [ encodeSeconds "down" seconds
                    , attribute "start" "true"
                    , on "stop" decodeStop
                    ]

                CountUp seconds ->
                    [ attribute "start" "true"
                    , encodeSeconds "up" seconds
                    , on "stop" decodeStop
                    ]
        ]


counter : List (Html.Attribute msg) -> Html msg
counter attributes =
    node "custom-counter" attributes []


encodeSeconds : String -> Seconds -> Html.Attribute msg
encodeSeconds direction seconds =
    [ ( "direction", Json.Encode.string direction )
    , ( "seconds", Json.Encode.float seconds )
    ]
        |> Json.Encode.object
        |> property "count"


decodeStop : Json.Decode.Decoder Msg
decodeStop =
    Json.Decode.map StopCount
        (Json.Decode.at [ "detail", "direction" ] directionDecoder)


directionDecoder : Json.Decode.Decoder Direction
directionDecoder =
    Json.Decode.string
        |> Json.Decode.andThen
            (\dir ->
                case dir of
                    "up" ->
                        Json.Decode.succeed Up

                    "down" ->
                        Json.Decode.succeed Down

                    _ ->
                        Json.Decode.fail <| "Unknown direction '" ++ dir ++ "'"
            )


main : Program () Model Msg
main =
    Browser.sandbox
        { init = Choose
        , view = view
        , update = update
        }
