port module Main exposing (main)

import Browser
import Browser.Events
import Char
import Html exposing (..)
import Html.Attributes exposing (..)
import Html.Events exposing (onClick)
import Html.Keyed
import Json.Decode
import Random
import Random.List
import Task
import Time



---- MODEL ----


type alias Model =
    { view : View
    , score : Int
    , highscore : Int
    , questions : List ( String, Int )
    , countdown : Int
    , level : Int
    , counter : Int
    }


type View
    = Start
    | Question String Int
    | End String Int



---- UPDATE ----


type Msg
    = StartGame
    | NextQuestion
    | KeyPress String
    | Choose Int
    | StartLevel Int (List ( String, Int ))
    | Tick Time.Posix
    | CountDown Time.Posix


addSeconds : Int -> Time.Posix -> Int
addSeconds seconds posix =
    Time.posixToMillis posix + (1000 * seconds)


update : Msg -> Model -> ( Model, Cmd Msg )
update msg model =
    case msg of
        StartGame ->
            startGame model

        StartLevel level questions ->
            { model
                | level = level
                , questions =
                    List.take
                        (numberOfQuestionsForLevel level)
                        questions
            }
                |> nextQuestion

        NextQuestion ->
            nextQuestion model

        CountDown time ->
            ( { model
                | countdown = addSeconds 3 time
              }
            , Cmd.none
            )

        Choose int ->
            checkValid int model

        KeyPress key ->
            case model.view of
                Question _ _ ->
                    case String.toInt key of
                        Just number ->
                            checkValid number model

                        _ ->
                            ( model, Cmd.none )

                _ ->
                    if key == "Enter" then
                        startGame model

                    else
                        ( model, Cmd.none )

        Tick posix ->
            case model.view of
                Question _ _ ->
                    let
                        time =
                            Time.posixToMillis posix
                    in
                    if time <= model.countdown then
                        ( model, Cmd.none )

                    else
                        checkValid -1 model

                _ ->
                    ( model, Cmd.none )


startGame : Model -> ( Model, Cmd Msg )
startGame model =
    { model
        | questions = []
        , score = 0
        , level = 0
        , countdown = 5507546605289
    }
        |> nextQuestion


nextQuestion : Model -> ( Model, Cmd Msg )
nextQuestion model =
    case model.questions of
        [] ->
            let
                level =
                    model.level + 1
            in
            -- next level
            ( model
            , Random.generate (StartLevel level) <|
                Random.List.shuffle (calculateQuestions level)
            )

        ( question, answer ) :: rest ->
            ( { model
                | view = Question question answer
                , questions = rest
                , counter = model.counter + 1
              }
            , Task.perform CountDown Time.now
            )


calculateQuestions : Int -> List ( String, number )
calculateQuestions level =
    calc [] (level + 1)
        |> List.filterMap filterSums
        |> assertMinLength (numberOfQuestionsForLevel level)


calc : List Int -> Int -> List (List Int)
calc acc iterations =
    if iterations == 1 then
        [ 1 :: acc, 2 :: acc, 3 :: acc ]

    else
        [ 1, 2, 3, -1, -2, -3 ]
            |> List.map (\i -> calc (i :: acc) (iterations - 1))
            |> List.concat


numberOfQuestionsForLevel : Int -> Int
numberOfQuestionsForLevel level =
    (level + 1) * 5


assertMinLength : Int -> List a -> List a
assertMinLength length list =
    if List.length list < length then
        assertMinLength length (list ++ list)

    else
        list


filterSums : List Int -> Maybe ( String, number )
filterSums list =
    case List.sum list of
        1 ->
            Just ( equationToString "" list, 1 )

        2 ->
            Just ( equationToString "" list, 2 )

        3 ->
            Just ( equationToString "" list, 3 )

        _ ->
            Nothing


equationToString : String -> List Int -> String
equationToString acc list =
    case list of
        first :: rest ->
            equationToString (acc ++ asString first) rest

        [] ->
            String.dropLeft 1 acc


asString : Int -> String
asString int =
    if int < 0 then
        String.fromInt int

    else
        "+" ++ String.fromInt int


checkValid : Int -> Model -> ( Model, Cmd Msg )
checkValid answer model =
    case model.view of
        Question question expected ->
            if answer == expected then
                { model | score = model.score + model.level }
                    |> nextQuestion

            else
                let
                    highscore =
                        if model.score > model.highscore then
                            model.score

                        else
                            model.highscore
                in
                ( { model
                    | view = End question answer
                    , highscore = highscore
                  }
                , storeHighscore highscore
                )

        _ ->
            ( model, Cmd.none )



---- VIEW ----


view : Model -> Browser.Document Msg
view model =
    { title = "123 game"
    , body =
        [ div [ class "centered" ] <|
            case model.view of
                Start ->
                    [ h2 [] [ text "3 - 1 = ?" ]
                    , p [] [ text "You have 3 seconds to solve each equation." ]
                    , p [] [ text "This game stores your highscores on your machine using localStorage" ]
                    , renderHighscore model
                    , button [ onClick StartGame ] [ text "Start the game" ]
                    ]

                Question question _ ->
                    [ p [ class "score" ]
                        [ text <| "Score: " ++ String.fromInt model.score ]
                    , h1 [] [ text <| question ++ " = " ]
                    , Html.Keyed.ul [] <|
                        List.map (btn model.counter) [ 1, 2, 3 ]
                    ]

                End question answer ->
                    [ h1 [] [ text <| question ++ formatWrongAnswer answer ]
                    , h2 [] [ text <| "Score: " ++ String.fromInt model.score ]
                    , renderHighscore model
                    , button [ onClick StartGame ] [ text "Play again?" ]
                    ]
        ]
    }


renderHighscore : Model -> Html Msg
renderHighscore { highscore } =
    h3 [] [ text <| "Personal best: " ++ String.fromInt highscore ]


btn : Int -> Int -> ( String, Html Msg )
btn counter int =
    ( "btn" ++ String.fromInt int ++ "-" ++ String.fromInt counter
    , div
        [ class "button", onClick <| Choose int ]
        [ text <| String.fromInt int
        , div [ class <| "countdown" ] []
        , div [ class "center" ] [ text <| String.fromInt int ]
        ]
    )


formatWrongAnswer : Int -> String
formatWrongAnswer answer =
    " " ++ String.fromChar (Char.fromCode 8800) ++ " " ++ String.fromInt answer



---- PROGRAM ----


main : Program Flags Model Msg
main =
    Browser.document
        { view = view
        , init = init
        , update = update
        , subscriptions = subscriptions
        }


type alias Flags =
    { highscore : Int
    }


init : Flags -> ( Model, Cmd Msg )
init flags =
    ( { view = Start
      , score = 0
      , highscore = flags.highscore
      , questions = []
      , level = 0
      , countdown = 0
      , counter = 0
      }
    , Cmd.none
    )



---- SUBSCRIPTIONS ----


subscriptions : Model -> Sub Msg
subscriptions model =
    Sub.batch
        [ Browser.Events.onKeyDown (Json.Decode.map KeyPress keyDecoder)
        , case model.view of
            Question _ _ ->
                Time.every 500 Tick

            _ ->
                Sub.none
        ]


keyDecoder : Json.Decode.Decoder String
keyDecoder =
    Json.Decode.field "key" Json.Decode.string



---- PORTS ----


port storeHighscore : Int -> Cmd msg
