﻿"use strict"

var Region = require('./region.js');

var WaterSolver = function _WaterSolver(rainHours, landscape) {
    var self = this;
    initializeValues();
    
    function initializeValues() {
        if (typeof landscape === 'undefined') {
            console.log('rainHours:', rainHours);
            throw new Error('landscape is undefined');
        }
        
        self.result = [];
        self.rainHours = rainHours;
        self.initialLandscape = landscape;
        self.border = new Region.Border(landscape.length);
        self.groupedByHeight = {};
        self.maxHeight = 0;
    }
    
    self.calculate = function _calculate() {
        // if the landscape overflows completely, it returns a number > 0
        var solved = initializeLandscape();
        
        if (solved > 0) {
            // the returned number equals the height of each column
            for (let i = landscape.length; i > 0; i--) self.result.push(solved);
            return;
        } else {
            fillLandscapeWithRain();
            let region = self.border.right;
            while (region !== self.border) {
                for (let i = 0; i < region.width(); i++) {
                    self.result.push(region.height);
                }
                region = region.right;
            }
        }
    }
    
    
    /**
     * This reads the provided landscape and creates a double-linked list of regions.
     * The local minima and maxima are selected and the rain is diverted to the local minima.
     * Also the regions are grouped by their height
     */
    function initializeLandscape() {
        var start = 0, end,
            height = landscape[start], summedHeight = 0,
            region = null;
        
        createRegionsArray();
        // check if overflown
        var capacity = self.maxHeight * landscape.length - summedHeight;
        var rainAmount = rainHours * landscape.length;
        if (rainAmount > capacity) {
            // landscape is totally overflown, return the height of each column
            return self.maxHeight + (rainAmount - capacity) / landscape.length;
        } else {
            moveRainToLocalMinima();
            return 0;
        }
        
        
        function createRegionsArray() {
            var lastRegion = self.border, lastMinimum = null, lastMaximum = self.border;
            
            for (end = 1; end < landscape.length; end++) {
                if (landscape[end] !== height) {
                    // add new region
                    region = new Region(start, end - 1, height, lastRegion);
                    region.rain = rainHours * region.width();
                    addToGroup(region);
                    if (height > self.maxHeight) self.maxHeight = height;
                    summedHeight += height * region.width();
                    lastRegion = region;
                    start = end;
                    height = landscape[end];
                }
            }
            region = new Region(start, end - 1, height, lastRegion, self.border);
            region.rain = rainHours * region.width();
            addToGroup(region);
            summedHeight += height * region.width();
        }
        
        // all rain will be distributed into the local minima
        function moveRainToLocalMinima() {
            var region = self.border.right;
            var lastMinimum = null;
            
            while (region !== self.border) {
                switch (region.type()) {
                    case Region.TYPES.LOCAL_MAXIMUM:
                        region.pourInto(lastMinimum, region.right);
                        break;
                    case Region.TYPES.LOCAL_MINIMUM:
                        region.left.pourInto(region);
                        lastMinimum = region;
                        break;
                    case Region.TYPES.ASCENDING:
                        region.pourInto(lastMinimum);
                        break;
                    case Region.TYPES.DESCENDING:
                        region.pourInto(region.right);
                        break;
                }
                region = region.right;
            }
        }
    }
    
    /**
     * This function fills the landscape with rain from the bottom up
     */
    function fillLandscapeWithRain() {
        var heightIndex = 0;
        
        while (heightIndex >= 0 && heightIndex <= self.maxHeight) {
            let group = self.groupedByHeight[heightIndex];
            // if the passed index does not exist, find the next existing index and try again
            if (typeof group === 'undefined' || !Array.isArray(group)) {
                let indizes = Object.keys(self.groupedByHeight);
                for (let i = 0; i < indizes.length; i++) {
                    if (indizes[i] > heightIndex) {
                        heightIndex = indizes[i];
                        break;
                    }
                }
                continue;
            }
            
            delete self.groupedByHeight[heightIndex++];
            group.forEach(function (region) {
                // this region was deleted, skip it
                if (typeof region.deleted !== 'undefined' || region.deleted) return;

                if (region.hasRain()) heightIndex = fillRegion(region);
                addToGroup(region);
            });
        }
    }
    
    function fillRegion(region) {
        region.fill();
        // merge with neighbors of same height level
        region.mergeThisHeight('left');
        region.mergeThisHeight('right');
        var oldHeight = Math.floor(region.height);
        //fill
        region.fill();
        
        if (!region.isOverflowing()) return 0;

        // try to pour rain downwards if possible
        switch (region.type()) {
            case Region.TYPES.DESCENDING:
                return region.pourIntoNextMinimum('right');
            case Region.TYPES.ASCENDING:
                return region.pourIntoNextMinimum('left');
            default:
                // merge this region with a higher region
                let newHeight = Math.floor(region.height);
                if (oldHeight !== newHeight) break;
                // same height as before ? -> merge with next higher level
                region.mergeUpwards();
                region.fill();
                return Math.floor(region.height);
        }
        return oldHeight;
    }
    
    function addToGroup(region) {
        var height = Math.floor(region.height);
        if (typeof self.groupedByHeight[height] === 'undefined') self.groupedByHeight[height] = [region]
        else self.groupedByHeight[height].push(region);
        if (height > self.maxHeight) self.maxHeight = height;
    }
}

WaterSolver.prototype.toString = function (verbose) {
    var str = '';
    if (!this.result.length > 0) {
        str = 'It will rain for ' + this.rainHours + ' hours on [' + this.initialLandscape.join(', ') + ']';
    } else {
        str = 'It has rained for ' + this.rainHours + ' hours on [' + this.initialLandscape.join(', ') + '], result is [' + this.result.join(', ') + ']';
        if (verbose) {
            let region = this.border.right;
            str += '\nregions:';
            while (region !== this.border && typeof region !== 'undefined') {
                str += '\n  ' + region.toString();
                region = region.right;
            }
        }
    }
    return str;
}


module.exports = {
    Solver: WaterSolver,
    Types: Region.TYPES
}
