﻿"use strict"

var TYPES = {
    UNKNOWN: -1,
    DESCENDING: 0,
    LOCAL_MINIMUM: 1,
    LOCAL_MAXIMUM: 2,
    ASCENDING: 3,
    BORDER: 4,
    
    toString: function (key) {
        var keys = Object.keys(TYPES);
        for (let i in keys) {
            if (TYPES[keys[i]] === key) return keys[i];
        }
        return 'TYPE:' + key;
    }
}

function Region(startIndex, endIndex, height, left, right) {
    this.startIndex = startIndex;
    this.endIndex = endIndex;
    this.height = height;
    this.left = left;
    if (typeof left !== 'undefined') left.right = this;
    this.right = right;
    if (typeof right !== 'undefined') right.left = this;
    
    this.rain = 0;
}

Region.prototype.isRegion = true;
Region.prototype.hasRain = function () { return this.rain > 0; }
Region.prototype.isOverflowing = function () { return this.hasRain() && this.capacity() <= 0; }

Region.prototype.toString = function () {
    var first = (this.deleted) ? 'deleted' : 'Region';
    return first + ':' + TYPES.toString(this.type()) + ' { start: ' + this.startIndex + ', end: ' + this.endIndex + ', width: ' + this.width() + 
    ', height: ' + this.height + ', rain: ' + this.rain + ' }';
}

Region.prototype.type = function _type() {
    var type;
    try {
        if (this.left.height > this.height) {
            if (this.height > this.right.height) type = TYPES.DESCENDING;
            else type = TYPES.LOCAL_MINIMUM;
        } else {
            if (this.height > this.right.height) type = TYPES.LOCAL_MAXIMUM;
            else type = TYPES.ASCENDING;
        }
    } catch (ex) {
        type = TYPES.UNKNOWN;
    }
    return type;
}

Region.prototype.width = function _width() {
    return this.endIndex - this.startIndex + 1;
}

Region.prototype.capacity = function _capacity() {
    var lowerNeighborHeight = Math.min(this.left.height, this.right.height);
    var width = this.width();
    return width * (lowerNeighborHeight - this.height);
}

Region.prototype.pourInto = function _pour(otherRegion, amount) {
    if (typeof amount !== 'undefined' && amount.isRegion) {
        // in this case the rain is split evenly between two regions
        let secondDestination = amount;
        amount = this.rain / 2;
        this.pourInto(secondDestination, amount);
    }
    if (typeof amount === 'undefined' || amount > this.rain) amount = this.rain;
    
    otherRegion.rain += amount;
    this.rain -= amount;
}

Region.prototype.pourIntoNextMinimum = function (direction) {
    var dest = this.getNextMinimum(direction);
    this.pourInto(dest);
    return Math.floor(dest.height);
}

Region.prototype.getNextMinimum = function _findMinimum(direction) {
    var dest = this[direction];
    // find next local minimum
    while (dest.type() !== Region.TYPES.LOCAL_MINIMUM) {
        dest = dest[direction];
    }
    return dest;
}

Region.prototype.fill = function _fill() {
    var amount = Math.min(this.capacity(), this.rain);
    if (amount > 0) {
        this.height += amount / this.width();
        this.rain -= amount;
    }
}

Region.prototype.mergeUpwards = function _mergeUp(otherRegion) {
    if (!this.isOverflowing()) throw new ArgumentError('Region is not overflowing, no merge allowed');
    if (otherRegion.deleted) return; // this region was marked as deleted, don't use it
    
    var diff = this.left.height - this.right.height;
    if (diff == 0) {
        this.mergeWith(this.left);
        this.mergeWith(this.right);
    } else {
        let direction = (diff > 0) ? 'left' : 'right';
        this.mergeWith(this[direction]);
    }
}

/**
 * @param direction: 'left' or 'right'
 */
Region.prototype.mergeThisHeight = function (direction) {
    if (this[direction].height === this.height) {
        this.mergeWith(this[direction]);
        
        return this.mergeThisHeight(direction);
    }
}

Region.prototype.mergeWith = function _merge(otherRegion) {
    this.startIndex = Math.min(this.startIndex, otherRegion.startIndex);
    this.endIndex = Math.max(this.endIndex, otherRegion.endIndex);
    
    otherRegion.pourInto(this);
    otherRegion.markAsDeleted();
    
    if (this.left === otherRegion) {
        this.left = otherRegion.left;
        this.left.right = this;
    }
    if (this.right === otherRegion) {
        this.right = otherRegion.right;
        this.right.left = this;
    }
}

Region.prototype.markAsDeleted = function _delete() {
    this.deleted = true;
}

/**
 * a special kind of region, with height of infinity (to the left and right of the passed landscape)
 */
function Border(start) {
    var border = new Region(start, -1, Number.POSITIVE_INFINITY);
    border.type = function () { return Region.TYPES.BORDER; };
    border.width = function () { return 0; };
    return border;
}


Region.TYPES = TYPES;
Region.Border = Border;
module.exports = Region;
